#!/bin/sh

# Bash usually sets the exit code to the last command in a pipe stream.
# Setting pipefail tells bash to set the exit code to the last failing code in
# a pipe stream.  I need this for (cmd | tee) to return the exit code of cmd.

# Unfortunately, pipefail is not supported on the version of bash.exe
# distributed with the currently packaged version of Cygwin (Dec 2004).
#set -o pipefail

PATH=/bin:/usr/bin:/sbin:/usr/sbin
TEEOPTS=

# Assume this script is executed as "install/helper.sh"

cmd() {
  echo
  echo "\$ $@"
  echo
  "$@"
}

logcmd() {
  LOG="$1"
  shift
  cmd "$@" 2>&1 | tee $TEEOPTS "$LOG"
}

logcmda() {
  TEEOPTS=-a logcmd "$@"
}

banner() {
  echo "=="
  echo "==  $1"
  echo "=="
  echo
}

pauseOnError() {
  if [ $? != 0 ]
  then
    echo
    echo "*** Error. Press ENTER to continue. ***"
    read a
    false
  fi
}

verify_rpms_exist() {
  ec=0
  for a in `cat "$@"`; do
    if [ ! -f "$a" ]; then
      echo "File not found: $a";
      ec=1
    fi
  done
  [ $ec = 0 ]
}

uninstall_list() {
  perl -pe 's{.*/}{}; s{\.[^.]+\.[^.]+\.rpm$}{};' "$@"
}

uninstall_nesc_1p1() {
  NC11="`rpm -qa nesc | grep '^nesc-1\.1\.'`"
  [ x"$NC11" = x"" ] || cmd rpm --erase --nodeps $NC11
}

BASE_LIST1=install/windows-tinyos-base-rpms.txt
BASE_LIST2=install/windows-tinyos-base-tinyos-rpm.txt
BASE_LOGI=/tmp/moteiv-tmote-tools-tinyos-base.log
BASE_LOGU=/tmp/moteiv-tmote-tools-tinyos-base-uninstall.log

MOTEIV_LIST=install/windows-tinyos-moteiv-rpms.txt
MOTEIV_LOGI=/tmp/moteiv-tmote-tools-tinyos-moteiv.log
MOTEIV_LOGU=/tmp/moteiv-tmote-tools-tinyos-moteiv-uninstall.log

[ -d /tmp ] || mkdir /tmp

case $1 in

  install-windows-tinyos-base)
    banner "Installing TinyOS Base RPM's"

    verify_rpms_exist $BASE_LIST1 $BASE_LIST2 && \
    logcmd $BASE_LOGI uninstall_nesc_1p1 && \
    logcmda $BASE_LOGI rpm -ivh --force --ignoreos `cat $BASE_LIST1` && \
    logcmda $BASE_LOGI rpm -ivh --force --nodeps --ignoreos `cat $BASE_LIST2`

    ;;

  uninstall-windows-tinyos-base)
    banner "Uninstalling TinyOS Base RPM's"

    logcmd $BASE_LOGU rpm --erase -v --nodeps `uninstall_list $BASE_LIST2 $BASE_LIST1`

    ;;

  install-windows-tinyos-moteiv)
    banner "Installing Moteiv RPM's"

    verify_rpms_exist $MOTEIV_LIST && \
    logcmd $MOTEIV_LOGI rpm -ivh --force --nodeps --ignoreos `cat $MOTEIV_LIST`

    ;;

  uninstall-windows-tinyos-moteiv)
    banner "Uninstalling Moteiv RPM's"

    logcmd $MOTEIV_LOGU rpm --erase -v `uninstall_list $MOTEIV_LIST`

    ;;

  fail)
    banner 'Failing by request; command "fail" given.'
    false
    ;;

  *)
    cat<<EOF

usage: helper.sh [command] (media_root_dir)

  Commands:

    install-windows-tinyos-base
    install-windows-tinyos-moteiv

    uninstall-windows-tinyos-base
    uninstall-windows-tinyos-moteiv

EOF
    [ x$1 = x ]  # fail if first argument was not empty
    ;;


esac

pauseOnError

